package easik.sketch.datatype;



import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import easik.Easik;
import easik.EasikConstants;
import easik.sketch.Sketch;
import easik.sketch.attribute.EntityAttribute;
import easik.sketch.util.Export.ExportConstants;
import easik.sketch.vertex.EntityNode;


/**
 * Class to controll data type information.  Supplies relevant datatypes and information 
 * depending on user defined data type group.
 * 
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @version 2006-08-23 Kevin Green
 */
public class DataTypeController {
	
	/**
	 * The data types defined for this data type controller
	 */
	private ArrayList<DataType> _dataTypes;
	/**
	 * Stores whether MySQL types are active
	 */
	private boolean _useMySQL = true;
	/**
	 * Stores whether Oracle types are active
	 */
	private boolean _useOracle = false;
	/**
	 * Stores whether DB2 types are active
	 */
	private boolean _useDB2 = false;
	/**
	 * Stores whether XML types are active
	 */
	private boolean _useXML = false;
	/**
	 * Stores whether User Defined types are active
	 */
	private boolean _useUserDefined = false;
	
	/**
	 * Default Constructor
	 *
	 */
	public DataTypeController(){
		_dataTypes = new ArrayList<DataType>();
		Iterator it = EasikConstants.getDefaultDataTypes().keySet().iterator();
		while(it.hasNext()){
			DataType curType = (DataType) EasikConstants.getDefaultDataTypes().get(it.next());
			_dataTypes.add(curType);
		}
	}
	
	/**
	 * Returns an ArrayList of the current data types available for the
	 * data type platform used by the sketch
	 * 
	 * @return ArrayList of the current data types available for the data type platform used by the sketch
	 */
	public ArrayList getDataTypes(){
		return _dataTypes;
	}
	
	/**
	 * Searches through data types for types named the same as standard types
	 * and loads up the standard value into the null position. If no name match
	 * is found then it loads the default value into the null position.
	 * 
	 * @param type The platform that is being loaded from nulls (If no specified from given platforms then only null values are reloaded)
	 */
	public void loadNullsFromDefaults(String type){
		int numTypes = _dataTypes.size();
		for(int i=0; i<numTypes; i++){
			DataType curType = (DataType)_dataTypes.get(i);
			if(EasikConstants.getDefaultDataTypes().containsKey(curType.getTypeName())){
				DataType standardMatch = (DataType) EasikConstants.getDefaultDataTypes().get(curType.getTypeName());
				if(type.equals("MySQL"))
					curType.set_MySQL_type(standardMatch.get_MySQL_type());
				else if(type.equals("Oracle"))
					curType.set_Oracle_type(standardMatch.get_Oracle_type());
				else if(type.equals("DB2"))
					curType.set_DB2_type(standardMatch.get_DB2_type());
				else if(type.equals("XML"))
					curType.set_XML_type(standardMatch.get_XML_type());
				else if(type.equals("UserDefined"))
					curType.set_user_type(curType.getTypeName());
			}
			else{
				if(curType.get_MySQL_type() == null || curType.get_MySQL_type().equals("")){
					curType.set_MySQL_type(EasikConstants.MY_SQL_DEFAULT);
				}
				if(curType.get_Oracle_type() == null || curType.get_Oracle_type().equals("")){
					curType.set_Oracle_type(EasikConstants.ORACLE_DEFAULT);				
				}
				if(curType.get_DB2_type() == null || curType.get_DB2_type().equals("")){
					curType.set_DB2_type(EasikConstants.DB2_DEFAULT);
				}
				if(curType.get_XML_type() == null || curType.get_XML_type().equals("")){
					curType.set_XML_type(EasikConstants.XML_DEFAULT);
				}
				if(curType.get_user_type() == null || curType.get_user_type().equals("")){
					curType.set_user_type(curType.getTypeName());
				}
			}
		}
	}
	
	/**
	 * Resets the data type array to the standard types
	 */
	public void resetDataTypes(){
		_dataTypes = new ArrayList<DataType>();
		HashMap tempDataTypes = EasikConstants.getDefaultDataTypes();
		Iterator it = EasikConstants.getDefaultDataTypes().keySet().iterator();
		while(it.hasNext()){
			DataType curType = (DataType) tempDataTypes.get(it.next());
			_dataTypes.add(curType);
		}


		Sketch mySketch = Easik.getInstance().getFrame().getSketch();
		DataType repDefault = (DataType) tempDataTypes.get("VarChar(255)");
		
		//Loop through all Entities
		Iterator it2 = mySketch.getEntities().keySet().iterator();
		while(it2.hasNext()){
			EntityNode curEntity = (EntityNode) mySketch.getEntities().get(it2.next());
			//Loop through attributes of entity
			int numAtts = curEntity.getAttributes().size();
			for(int i=0; i<numAtts; i++){
				EntityAttribute curAtt = (EntityAttribute) curEntity.getAttributes().get(i);
				curAtt.setDataType(repDefault);
				Easik.getInstance().getFrame().getInfoTreeUI().refreshTree(curAtt.getNode());
			}
		}
	}
	
	/**
	 * Adds a new data type entry to the _dataTypes ArrayList, and returns it
	 * @return the new DataType added to the list of datatypes of this controller
	 */
	public DataType addNewDataType(){
		int i = 1;
		while(!isNameFree(null, "New Data Type " + i))
			i++;
		
		String newName = "New Data Type " + i;
		
		DataType newType = new DataType(newName, "", 
				EasikConstants.MY_SQL_DEFAULT,
				EasikConstants.ORACLE_DEFAULT,
				EasikConstants.DB2_DEFAULT,
				EasikConstants.XML_DEFAULT,
				newName);
		_dataTypes.add(newType);
		return newType;
	}
	
	/**
	 * Adds a new data type entry to the _dataTypes ArrayList
	 * 
	 * @param inType The data type to be added.
	 */
	public void addDataType(DataType inType){
		_dataTypes.add(inType);
	}
	
	/**
	 * Removes the data type from the array. Replaces all references
	 * of the removed data type with the replacement data type.
	 * 
	 * @param inType The data type to be removed.
	 * @param inReplacement The data type to be used as the replacement
	 */
	public void removeDataType(DataType inType, DataType inReplacement){
		_dataTypes.remove(inType);
		Sketch mySketch = Easik.getInstance().getFrame().getSketch();
		
		//Loop through all Entities
		Iterator it = mySketch.getEntities().keySet().iterator();
		while(it.hasNext()){
			EntityNode curEntity = (EntityNode) mySketch.getEntities().get(it.next());
			//Loop through attributes of entity
			int numAtts = curEntity.getAttributes().size();
			for(int i=0; i<numAtts; i++){
				EntityAttribute curAtt = (EntityAttribute) curEntity.getAttributes().get(i);
				if(curAtt.getDataType().equals(inType)){
					curAtt.setDataType(inReplacement);
					Easik.getInstance().getFrame().getInfoTreeUI().refreshTree(curAtt.getNode());
				}
			}
		}
		
	}
	/**
	 * Checks to see if the data type name is used by any other
	 * data type other then itself
	 * 
	 * @param inType The type to compare
	 * @param newName The desired new name of the type
	 * @return True if the new name is free, false otherwise.
	 */
	public boolean isNameFree(DataType inType, String newName){
		int numTypes = _dataTypes.size();
		for(int i=0; i<numTypes; i++){
			DataType curType = (DataType)_dataTypes.get(i);
			if(curType != inType && curType.getTypeName().toUpperCase().equals(newName.toUpperCase()))
				return false;
		}
		return true;
	}
	
	/**
	 * @return Returns the _useDB2.
	 */
	public boolean is_useDB2() {
		return _useDB2;
	}

	/**
	 * @param _usedb2 The _useDB2 to set.
	 */
	public void set_useDB2(boolean _usedb2) {
		_useDB2 = _usedb2;
	}

	/**
	 * @return Returns the _useMySQL.
	 */
	public boolean is_useMySQL() {
		return _useMySQL;
	}

	/**
	 * @param mySQL The _useMySQL to set.
	 */
	public void set_useMySQL(boolean mySQL) {
		_useMySQL = mySQL;
	}

	/**
	 * @return Returns the _useOracle.
	 */
	public boolean is_useOracle() {
		return _useOracle;
	}

	/**
	 * @param oracle The _useOracle to set.
	 */
	public void set_useOracle(boolean oracle) {
		_useOracle = oracle;
	}

	/**
	 * @return Returns the _useUserDefined.
	 */
	public boolean is_useUserDefined() {
		return _useUserDefined;
	}

	/**
	 * @param userDefined The _useUserDefined to set.
	 */
	public void set_useUserDefined(boolean userDefined) {
		_useUserDefined = userDefined;
	}

	/**
	 * @return Returns the _useXML.
	 */
	public boolean is_useXML() {
		return _useXML;
	}

	/**
	 * @param _usexml The _useXML to set.
	 */
	public void set_useXML(boolean _usexml) {
		_useXML = _usexml;
	}
	
	/**
	 * Gets a list of all types currently defined by the user.
	 * 
	 * @return An arraylist containing the names of defined types.
	 * @since 2006-06-27 Vera Ranieri
	 */
	public ArrayList getActiveTypes(){
		ArrayList<String> types = new ArrayList<String>();
		if(_useDB2)
			types.add(ExportConstants.DB2);
		if(_useMySQL)
			types.add(ExportConstants.MY_SQL);
		if(_useOracle)
			types.add(ExportConstants.ORACLE);
		if(_useXML)
			types.add(ExportConstants.XML);
		if(_useUserDefined)
			types.add(ExportConstants.USER_DEFINED);
		
		return types;
	}
}
